"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = _default;
var _math = require("../../../math");
var _utils = require("../../../utils");
/**
 * Builds a line to draw
 *
 * Ignored from docs since it is not directly exposed.
 *
 * @ignore
 * @private
 * @param {InkPaint.WebGLGraphicsData} graphicsData - The graphics object containing all the necessary properties
 * @param {object} webGLData - an object containing all the webGL-specific information to create this shape
 * @param {object} webGLDataNativeLines - an object containing all the webGL-specific information to create nativeLines
 */
function _default(graphicsData, webGLData, webGLDataNativeLines) {
  if (graphicsData.nativeLines) {
    buildNativeLine(graphicsData, webGLDataNativeLines);
  } else {
    buildLine(graphicsData, webGLData);
  }
}

/**
 * Builds a line to draw using the poligon method.
 *
 * Ignored from docs since it is not directly exposed.
 *
 * @ignore
 * @private
 * @param {InkPaint.WebGLGraphicsData} graphicsData - The graphics object containing all the necessary properties
 * @param {object} webGLData - an object containing all the webGL-specific information to create this shape
 */
function buildLine(graphicsData, webGLData) {
  // TODO OPTIMISE!
  var points = graphicsData.points;
  if (points.length === 0) {
    return;
  }
  // if the line width is an odd number add 0.5 to align to a whole pixel
  // commenting this out fixes #711 and #1620
  // if (graphicsData.lineWidth%2)
  // {
  //     for (i = 0; i < points.length; i++)
  //     {
  //         points[i] += 0.5;
  //     }
  // }

  // get first and last point.. figure out the middle!
  var firstPoint = new _math.Point(points[0], points[1]);
  var lastPoint = new _math.Point(points[points.length - 2], points[points.length - 1]);

  // if the first point is the last point - gonna have issues :)
  if (firstPoint.x === lastPoint.x && firstPoint.y === lastPoint.y) {
    // need to clone as we are going to slightly modify the shape..
    points = points.slice();
    points.pop();
    points.pop();
    lastPoint = new _math.Point(points[points.length - 2], points[points.length - 1]);
    var midPointX = lastPoint.x + (firstPoint.x - lastPoint.x) * 0.5;
    var midPointY = lastPoint.y + (firstPoint.y - lastPoint.y) * 0.5;
    points.unshift(midPointX, midPointY);
    points.push(midPointX, midPointY);
  }
  var verts = webGLData.points;
  var indices = webGLData.indices;
  var length = points.length / 2;
  var indexCount = points.length;
  var indexStart = verts.length / 6;

  // DRAW the Line
  var width = graphicsData.lineWidth / 2;

  // sort color
  var color = (0, _utils.hex2rgb)(graphicsData.lineColor);
  var alpha = graphicsData.lineAlpha;
  var r = color[0] * alpha;
  var g = color[1] * alpha;
  var b = color[2] * alpha;
  var p1x = points[0];
  var p1y = points[1];
  var p2x = points[2];
  var p2y = points[3];
  var p3x = 0;
  var p3y = 0;
  var perpx = -(p1y - p2y);
  var perpy = p1x - p2x;
  var perp2x = 0;
  var perp2y = 0;
  var perp3x = 0;
  var perp3y = 0;
  var dist = Math.sqrt(perpx * perpx + perpy * perpy);
  perpx /= dist;
  perpy /= dist;
  perpx *= width;
  perpy *= width;
  var ratio = graphicsData.lineAlignment; // 0.5;
  var r1 = (1 - ratio) * 2;
  var r2 = ratio * 2;

  // start
  verts.push(p1x - perpx * r1, p1y - perpy * r1, r, g, b, alpha);
  verts.push(p1x + perpx * r2, p1y + perpy * r2, r, g, b, alpha);
  for (var i = 1; i < length - 1; ++i) {
    p1x = points[(i - 1) * 2];
    p1y = points[(i - 1) * 2 + 1];
    p2x = points[i * 2];
    p2y = points[i * 2 + 1];
    p3x = points[(i + 1) * 2];
    p3y = points[(i + 1) * 2 + 1];
    perpx = -(p1y - p2y);
    perpy = p1x - p2x;
    dist = Math.sqrt(perpx * perpx + perpy * perpy);
    perpx /= dist;
    perpy /= dist;
    perpx *= width;
    perpy *= width;
    perp2x = -(p2y - p3y);
    perp2y = p2x - p3x;
    dist = Math.sqrt(perp2x * perp2x + perp2y * perp2y);
    perp2x /= dist;
    perp2y /= dist;
    perp2x *= width;
    perp2y *= width;
    var a1 = -perpy + p1y - (-perpy + p2y);
    var b1 = -perpx + p2x - (-perpx + p1x);
    var c1 = (-perpx + p1x) * (-perpy + p2y) - (-perpx + p2x) * (-perpy + p1y);
    var a2 = -perp2y + p3y - (-perp2y + p2y);
    var b2 = -perp2x + p2x - (-perp2x + p3x);
    var c2 = (-perp2x + p3x) * (-perp2y + p2y) - (-perp2x + p2x) * (-perp2y + p3y);
    var denom = a1 * b2 - a2 * b1;
    if (Math.abs(denom) < 0.1) {
      denom += 10.1;
      verts.push(p2x - perpx * r1, p2y - perpy * r1, r, g, b, alpha);
      verts.push(p2x + perpx * r2, p2y + perpy * r2, r, g, b, alpha);
      continue;
    }
    var px = (b1 * c2 - b2 * c1) / denom;
    var py = (a2 * c1 - a1 * c2) / denom;
    var pdist = (px - p2x) * (px - p2x) + (py - p2y) * (py - p2y);
    if (pdist > 196 * width * width) {
      perp3x = perpx - perp2x;
      perp3y = perpy - perp2y;
      dist = Math.sqrt(perp3x * perp3x + perp3y * perp3y);
      perp3x /= dist;
      perp3y /= dist;
      perp3x *= width;
      perp3y *= width;
      verts.push(p2x - perp3x * r1, p2y - perp3y * r1);
      verts.push(r, g, b, alpha);
      verts.push(p2x + perp3x * r2, p2y + perp3y * r2);
      verts.push(r, g, b, alpha);
      verts.push(p2x - perp3x * r2 * r1, p2y - perp3y * r1);
      verts.push(r, g, b, alpha);
      indexCount++;
    } else {
      verts.push(p2x + (px - p2x) * r1, p2y + (py - p2y) * r1);
      verts.push(r, g, b, alpha);
      verts.push(p2x - (px - p2x) * r2, p2y - (py - p2y) * r2);
      verts.push(r, g, b, alpha);
    }
  }
  p1x = points[(length - 2) * 2];
  p1y = points[(length - 2) * 2 + 1];
  p2x = points[(length - 1) * 2];
  p2y = points[(length - 1) * 2 + 1];
  perpx = -(p1y - p2y);
  perpy = p1x - p2x;
  dist = Math.sqrt(perpx * perpx + perpy * perpy);
  perpx /= dist;
  perpy /= dist;
  perpx *= width;
  perpy *= width;
  verts.push(p2x - perpx * r1, p2y - perpy * r1);
  verts.push(r, g, b, alpha);
  verts.push(p2x + perpx * r2, p2y + perpy * r2);
  verts.push(r, g, b, alpha);
  indices.push(indexStart);
  for (var _i = 0; _i < indexCount; ++_i) {
    indices.push(indexStart++);
  }
  indices.push(indexStart - 1);
}

/**
 * Builds a line to draw using the gl.drawArrays(gl.LINES) method
 *
 * Ignored from docs since it is not directly exposed.
 *
 * @ignore
 * @private
 * @param {InkPaint.WebGLGraphicsData} graphicsData - The graphics object containing all the necessary properties
 * @param {object} webGLData - an object containing all the webGL-specific information to create this shape
 */
function buildNativeLine(graphicsData, webGLData) {
  var i = 0;
  var points = graphicsData.points;
  if (points.length === 0) return;
  var verts = webGLData.points;
  var length = points.length / 2;

  // sort color
  var color = (0, _utils.hex2rgb)(graphicsData.lineColor);
  var alpha = graphicsData.lineAlpha;
  var r = color[0] * alpha;
  var g = color[1] * alpha;
  var b = color[2] * alpha;
  for (i = 1; i < length; i++) {
    var p1x = points[(i - 1) * 2];
    var p1y = points[(i - 1) * 2 + 1];
    var p2x = points[i * 2];
    var p2y = points[i * 2 + 1];
    verts.push(p1x, p1y);
    verts.push(r, g, b, alpha);
    verts.push(p2x, p2y);
    verts.push(r, g, b, alpha);
  }
}
//# sourceMappingURL=buildLine.js.map