"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
var _CanvasRenderer = _interopRequireDefault(require("../../renderers/canvas/CanvasRenderer"));
var _const = require("../../const");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
class CanvasGraphicsRenderer {
  constructor(renderer) {
    this.renderer = renderer;
  }
  render(graphics) {
    var renderer = this.renderer;
    var context = renderer.context;
    var worldAlpha = graphics.worldAlpha;
    var transform = graphics.transform.worldTransform;
    var resolution = renderer.resolution;
    context.setTransform(transform.a * resolution, transform.b * resolution, transform.c * resolution, transform.d * resolution, transform.tx * resolution, transform.ty * resolution);

    // update tint if graphics was dirty
    if (graphics.canvasTintDirty !== graphics.dirty || graphics._prevTint !== graphics.tint) {
      this.updateGraphicsTint(graphics);
    }
    renderer.setBlendMode(graphics.blendMode);
    for (var i = 0; i < graphics.graphicsData.length; i++) {
      var data = graphics.graphicsData[i];
      var shape = data.shape;
      var fillColor = data._fillTint;
      var lineColor = data._lineTint;
      context.lineWidth = data.lineWidth;
      if (data.type === _const.SHAPES.POLY) {
        context.beginPath();
        var points = shape.points;
        var holes = data.holes;
        var outerArea = void 0;
        var innerArea = void 0;
        var px = void 0;
        var py = void 0;
        context.moveTo(points[0], points[1]);
        for (var j = 2; j < points.length; j += 2) {
          context.lineTo(points[j], points[j + 1]);
        }

        // if the first and last point are the same close the path - much neater :)
        if (shape.closed) {
          context.closePath();
        }
        if (holes.length > 0) {
          outerArea = 0;
          px = points[0];
          py = points[1];
          for (var _j = 2; _j + 2 < points.length; _j += 2) {
            outerArea += (points[_j] - px) * (points[_j + 3] - py) - (points[_j + 2] - px) * (points[_j + 1] - py);
          }
          for (var k = 0; k < holes.length; k++) {
            points = holes[k].points;
            if (!points) {
              continue;
            }
            innerArea = 0;
            px = points[0];
            py = points[1];
            for (var _j2 = 2; _j2 + 2 < points.length; _j2 += 2) {
              innerArea += (points[_j2] - px) * (points[_j2 + 3] - py) - (points[_j2 + 2] - px) * (points[_j2 + 1] - py);
            }
            if (innerArea * outerArea < 0) {
              context.moveTo(points[0], points[1]);
              for (var _j3 = 2; _j3 < points.length; _j3 += 2) {
                context.lineTo(points[_j3], points[_j3 + 1]);
              }
            } else {
              context.moveTo(points[points.length - 2], points[points.length - 1]);
              for (var _j4 = points.length - 4; _j4 >= 0; _j4 -= 2) {
                context.lineTo(points[_j4], points[_j4 + 1]);
              }
            }
            if (holes[k].close) {
              context.closePath();
            }
          }
        }
        if (data.fill) {
          context.globalAlpha = data.fillAlpha * worldAlpha;
          context.fillStyle = "#" + ("00000" + (fillColor | 0).toString(16)).substr(-6);
          context.fill();
        }
        if (data.lineWidth) {
          context.globalAlpha = data.lineAlpha * worldAlpha;
          context.strokeStyle = "#" + ("00000" + (lineColor | 0).toString(16)).substr(-6);
          context.stroke();
        }
      } else if (data.type === _const.SHAPES.RECT) {
        if (data.fillColor || data.fillColor === 0) {
          context.globalAlpha = data.fillAlpha * worldAlpha;
          context.fillStyle = "#" + ("00000" + (fillColor | 0).toString(16)).substr(-6);
          context.fillRect(shape.x, shape.y, shape.width, shape.height);
        }
        if (data.lineWidth) {
          context.globalAlpha = data.lineAlpha * worldAlpha;
          context.strokeStyle = "#" + ("00000" + (lineColor | 0).toString(16)).substr(-6);
          context.strokeRect(shape.x, shape.y, shape.width, shape.height);
        }
      } else if (data.type === _const.SHAPES.CIRC) {
        // TODO - need to be Undefined!
        context.beginPath();
        context.arc(shape.x, shape.y, shape.radius, 0, 2 * Math.PI);
        context.closePath();
        if (data.fill) {
          context.globalAlpha = data.fillAlpha * worldAlpha;
          context.fillStyle = "#" + ("00000" + (fillColor | 0).toString(16)).substr(-6);
          context.fill();
        }
        if (data.lineWidth) {
          context.globalAlpha = data.lineAlpha * worldAlpha;
          context.strokeStyle = "#" + ("00000" + (lineColor | 0).toString(16)).substr(-6);
          context.stroke();
        }
      } else if (data.type === _const.SHAPES.ELIP) {
        // ellipse code taken from: http://stackoverflow.com/questions/2172798/how-to-draw-an-oval-in-html5-canvas

        var w = shape.width * 2;
        var h = shape.height * 2;
        var x = shape.x - w / 2;
        var y = shape.y - h / 2;
        context.beginPath();
        var kappa = 0.5522848;
        var ox = w / 2 * kappa; // control point offset horizontal
        var oy = h / 2 * kappa; // control point offset vertical
        var xe = x + w; // x-end
        var ye = y + h; // y-end
        var xm = x + w / 2; // x-middle
        var ym = y + h / 2; // y-middle

        context.moveTo(x, ym);
        context.bezierCurveTo(x, ym - oy, xm - ox, y, xm, y);
        context.bezierCurveTo(xm + ox, y, xe, ym - oy, xe, ym);
        context.bezierCurveTo(xe, ym + oy, xm + ox, ye, xm, ye);
        context.bezierCurveTo(xm - ox, ye, x, ym + oy, x, ym);
        context.closePath();
        if (data.fill) {
          context.globalAlpha = data.fillAlpha * worldAlpha;
          context.fillStyle = "#" + ("00000" + (fillColor | 0).toString(16)).substr(-6);
          context.fill();
        }
        if (data.lineWidth) {
          context.globalAlpha = data.lineAlpha * worldAlpha;
          context.strokeStyle = "#" + ("00000" + (lineColor | 0).toString(16)).substr(-6);
          context.stroke();
        }
      } else if (data.type === _const.SHAPES.RREC) {
        var rx = shape.x;
        var ry = shape.y;
        var width = shape.width;
        var height = shape.height;
        var radius = shape.radius;
        var maxRadius = Math.min(width, height) / 2 | 0;
        radius = radius > maxRadius ? maxRadius : radius;
        context.beginPath();
        context.moveTo(rx, ry + radius);
        context.lineTo(rx, ry + height - radius);
        context.quadraticCurveTo(rx, ry + height, rx + radius, ry + height);
        context.lineTo(rx + width - radius, ry + height);
        context.quadraticCurveTo(rx + width, ry + height, rx + width, ry + height - radius);
        context.lineTo(rx + width, ry + radius);
        context.quadraticCurveTo(rx + width, ry, rx + width - radius, ry);
        context.lineTo(rx + radius, ry);
        context.quadraticCurveTo(rx, ry, rx, ry + radius);
        context.closePath();
        if (data.fillColor || data.fillColor === 0) {
          context.globalAlpha = data.fillAlpha * worldAlpha;
          context.fillStyle = "#" + ("00000" + (fillColor | 0).toString(16)).substr(-6);
          context.fill();
        }
        if (data.lineWidth) {
          context.globalAlpha = data.lineAlpha * worldAlpha;
          context.strokeStyle = "#" + ("00000" + (lineColor | 0).toString(16)).substr(-6);
          context.stroke();
        }
      }
    }
  }

  /**
   * Updates the tint of a graphics object
   *
   * @private
   * @param {InkPaint.Graphics} graphics - the graphics that will have its tint updated
   */
  updateGraphicsTint(graphics) {
    graphics._prevTint = graphics.tint;
    graphics.canvasTintDirty = graphics.dirty;
    var tintR = (graphics.tint >> 16 & 0xff) / 255;
    var tintG = (graphics.tint >> 8 & 0xff) / 255;
    var tintB = (graphics.tint & 0xff) / 255;
    for (var i = 0; i < graphics.graphicsData.length; ++i) {
      var data = graphics.graphicsData[i];
      var fillColor = data.fillColor | 0;
      var lineColor = data.lineColor | 0;

      // super inline, cos optimization :)
      data._fillTint = ((fillColor >> 16 & 0xff) / 255 * tintR * 255 << 16) + ((fillColor >> 8 & 0xff) / 255 * tintG * 255 << 8) + (fillColor & 0xff) / 255 * tintB * 255;
      data._lineTint = ((lineColor >> 16 & 0xff) / 255 * tintR * 255 << 16) + ((lineColor >> 8 & 0xff) / 255 * tintG * 255 << 8) + (lineColor & 0xff) / 255 * tintB * 255;
    }
  }
  renderPolygon(points, close, context) {
    context.moveTo(points[0], points[1]);
    for (var j = 1; j < points.length / 2; ++j) {
      context.lineTo(points[j * 2], points[j * 2 + 1]);
    }
    if (close) {
      context.closePath();
    }
  }

  /**
   * destroy graphics object
   *
   */
  destroy() {
    this.renderer = null;
  }
}
exports.default = CanvasGraphicsRenderer;
_CanvasRenderer.default.registerPlugin("graphics", CanvasGraphicsRenderer);
//# sourceMappingURL=CanvasGraphicsRenderer.js.map